import { supabase, TABLES } from '@/config/supabase';
import { SubscriptionPlan, VendorSubscription, Feature, PlanFeature, TrialSettings } from './types';
import { getFeaturesForPlan } from '@/config/features';

export const subscriptionService = {
  // Subscription Plans
  async getAllPlans() {
    return supabase.from(TABLES.SUBSCRIPTION_PLANS).select('*').eq('is_active', true).order('price', { ascending: true });
  },
  async createPlan(plan: Partial<SubscriptionPlan>) {
    return supabase.from(TABLES.SUBSCRIPTION_PLANS).insert([plan]).select().single();
  },
  async updatePlan(id: string, plan: Partial<SubscriptionPlan>) {
    return supabase.from(TABLES.SUBSCRIPTION_PLANS).update(plan).eq('id', id).select().single();
  },
  async deletePlan(id: string) {
    // Soft delete
    return supabase.from(TABLES.SUBSCRIPTION_PLANS).update({ is_active: false }).eq('id', id);
  },

  // Vendor Subscriptions
  async getVendorSubscription(vendorId: string) {
    return supabase.from(TABLES.VENDOR_SUBSCRIPTIONS).select('*').eq('vendor_id', vendorId).order('created_at', { ascending: false }).limit(1).single();
  },
  async activateSubscription(vendorId: string, planId: string) {
    // Deactivate previous
    await supabase.from(TABLES.VENDOR_SUBSCRIPTIONS).update({ status: 'cancelled' }).eq('vendor_id', vendorId).eq('status', 'active');
    // Activate new
    return supabase.from(TABLES.VENDOR_SUBSCRIPTIONS).insert([{ vendor_id: vendorId, plan_id: planId, status: 'active', current_period_start: new Date().toISOString(), current_period_end: null }]).select().single();
  },
  async getTrialSettings() {
    return supabase.from(TABLES.TRIAL_SETTINGS).select('*').single();
  },
  async acceptTrial(vendorId: string, planId: string, durationDays: number) {
    const now = new Date();
    const trialEnd = new Date(now.getTime() + durationDays * 24 * 60 * 60 * 1000);
    return supabase.from(TABLES.VENDOR_SUBSCRIPTIONS).insert([{
      vendor_id: vendorId,
      plan_id: planId,
      status: 'trialing',
      trial_start: now.toISOString(),
      trial_end: trialEnd.toISOString(),
      current_period_start: now.toISOString(),
      current_period_end: trialEnd.toISOString()
    }]).select().single();
  },

  // Features
  async getAllFeatures() {
    return supabase.from(TABLES.FEATURES).select('*');
  },
  async updateFeature(id: string, data: Partial<Feature>) {
    return supabase.from(TABLES.FEATURES).update(data).eq('id', id).select().single();
  },
  async assignFeatureToPlan(planId: string, featureId: string) {
    return supabase.from(TABLES.SUBSCRIPTION_PLAN_FEATURES).insert([{ plan_id: planId, feature_id: featureId }]).select().single();
  },
  async removeFeatureFromPlan(planId: string, featureId: string) {
    return supabase.from(TABLES.SUBSCRIPTION_PLAN_FEATURES).delete().eq('plan_id', planId).eq('feature_id', featureId);
  },

  // Trial Settings
  async updateTrialSettings(data: Partial<TrialSettings>) {
    return supabase.from(TABLES.TRIAL_SETTINGS).update(data).eq('id', 1).select().single();
  },

  // Feature Gate: get available features for a user
  async getAvailableFeatures(userId: string) {
    // 1. Get vendor by userId
    const { data: vendor } = await supabase.from(TABLES.VENDORS).select('id').eq('user_id', userId).single();
    if (!vendor) return [];
    // 2. Get active subscription
    const { data: sub } = await supabase.from(TABLES.VENDOR_SUBSCRIPTIONS).select('plan_id,status').eq('vendor_id', vendor.id).eq('status', 'active').single();
    if (!sub) return [];
    // 3. Get plan
    const { data: plan } = await supabase.from(TABLES.SUBSCRIPTION_PLANS).select('name').eq('id', sub.plan_id).single();
    if (!plan) return [];
    // 4. Get features for plan
    return getFeaturesForPlan(plan.name);
  }
}; 